#include "dc1394dialog.h"
#include "ui_dc1394dialog.h"
#include <QDebug>
#include <iostream>

Dc1394Dialog::Dc1394Dialog(dc1394camera_t *camera, QWidget *parent)
    : QMainWindow(parent), ui(new Ui::Dc1394Dialog), camera(camera)
{
    ui->setupUi(this);

    bool initialized = initialize();
    qDebug() << "Initialize returned: " << initialized;
}

Dc1394Dialog::~Dc1394Dialog()
{
    delete ui;
    if (camera) dc1394_camera_free(camera);
}

#define CHECK_ERROR(err, clean, msg) if (err!=DC1394_SUCCESS) { std::cerr << msg << std::endl; clean ; return false; }
#define CHECK_ERROR(err, msg) if ((err)!=DC1394_SUCCESS) { std::cerr << msg << std::endl; }
#define CHECK_ERROR(err) if ((err)!=DC1394_SUCCESS) { std::cerr << #err << std::endl; }
bool Dc1394Dialog::initialize()
{
    if (!camera) {
        dc1394_t *d = dc1394_new ();
        if (!d)
            return false;
        dc1394camera_list_t * list;
        dc1394error_t err=dc1394_camera_enumerate (d, &list);

        if (list->num == 0) {
            dc1394_free (d);
            return false;
        }

        camera = dc1394_camera_new (d, list->ids[0].guid);
        dc1394_camera_free_list (list);
        if (!camera) {
            dc1394_log_error("Failed to initialize camera");
            dc1394_free (d);
            return false;
        }
    }

    dc1394_feature_get_all(camera, &features);
    dc1394_feature_print_all(&features, stderr);

    setupFeature("Gain", DC1394_FEATURE_GAIN, ui->gainSpin);
    setupFeature("Exposure", DC1394_FEATURE_EXPOSURE, ui->exposureSpin);
    setupFeature("Shutter", DC1394_FEATURE_SHUTTER, ui->shutterSpin);
    setupFeature("Brightness", DC1394_FEATURE_BRIGHTNESS, ui->brightnessSpin);

    timer = startTimer(2000);
    timerEvent(0);
    return true;
}

void Dc1394Dialog::setupFeature(QString name, dc1394feature_t feature, QDoubleSpinBox *spin)
{
    uint32_t vmin,vmax;
    dc1394_feature_get_boundaries(camera, feature, &vmin, &vmax);
    spin->setMaximum(vmax);
    spin->setMinimum(vmin);

    dc1394switch_t power;
    dc1394_feature_get_power(camera, feature, &power);
    qDebug() << name  << ", power: " << (power == DC1394_ON ? "on" : "off");
}

void Dc1394Dialog::timerEvent( QTimerEvent *)
{
    updateFeature(DC1394_FEATURE_GAIN, ui->gainSpin, ui->manualGain);
    updateFeature(DC1394_FEATURE_EXPOSURE, ui->exposureSpin, ui->manualExposure);
    updateFeature(DC1394_FEATURE_SHUTTER, ui->shutterSpin, ui->manualShutter);
    updateFeature(DC1394_FEATURE_BRIGHTNESS, ui->brightnessSpin, ui->manualBrightness);

    uint32_t u_b_val, u_r_val;
    dc1394_feature_whitebalance_get_value(camera, &u_b_val, &u_r_val);
    ui->whiteBalSliderBlue->setValue(u_b_val);
    ui->whiteBalSliderRed->setValue(u_r_val);

}

void Dc1394Dialog::updateFeature(dc1394feature_t feature, QDoubleSpinBox *spin, QCheckBox *checkbox)
{
    uint32_t val;
    CHECK_ERROR(dc1394_feature_get_value(camera, feature, &val));
    spin->setValue(val);
    dc1394feature_mode_t mode;
    CHECK_ERROR(dc1394_feature_get_mode(camera, feature, &mode));
    checkbox->setCheckState(mode == DC1394_FEATURE_MODE_AUTO ? Qt::Unchecked :Qt::Checked);
}

void Dc1394Dialog::on_autoGain_clicked()
{
    CHECK_ERROR(dc1394_feature_set_mode(camera, DC1394_FEATURE_GAIN, DC1394_FEATURE_MODE_ONE_PUSH_AUTO));
   // timerEvent(0);
}


void Dc1394Dialog::on_manualGain_toggled(bool checked)
{
    qDebug() << "gain mode:" << (checked ? "manual" : "auto");
    CHECK_ERROR(dc1394_feature_set_mode(camera, DC1394_FEATURE_GAIN,
                            (checked ? DC1394_FEATURE_MODE_MANUAL : DC1394_FEATURE_MODE_AUTO)));
    //timerEvent(0);
}

void Dc1394Dialog::on_gainSpin_valueChanged(int gain)
{
    CHECK_ERROR(dc1394_feature_set_value(camera, DC1394_FEATURE_GAIN, gain));
    //timerEvent(0);
}


void Dc1394Dialog::on_exposureSpin_valueChanged(double val)
{
    CHECK_ERROR(dc1394_feature_set_value(camera, DC1394_FEATURE_EXPOSURE, val));
    //timerEvent(0);
}

void Dc1394Dialog::on_autoExposure_clicked()
{
    CHECK_ERROR(dc1394_feature_set_mode(camera, DC1394_FEATURE_EXPOSURE, DC1394_FEATURE_MODE_ONE_PUSH_AUTO));
    //timerEvent(0);
}

void Dc1394Dialog::on_manualExposure_toggled(bool checked)
{
    CHECK_ERROR(dc1394_feature_set_mode(camera, DC1394_FEATURE_EXPOSURE,
                            (checked ? DC1394_FEATURE_MODE_MANUAL : DC1394_FEATURE_MODE_AUTO)));
    //timerEvent(0);
}

void Dc1394Dialog::on_manualWhiteBal_toggled(bool checked)
{
    CHECK_ERROR(dc1394_feature_set_mode(camera, DC1394_FEATURE_WHITE_BALANCE,
                            (checked ? DC1394_FEATURE_MODE_MANUAL : DC1394_FEATURE_MODE_AUTO)));
    //timerEvent(0);
}


void Dc1394Dialog::on_whiteBalSliderBlue_sliderMoved(int position)
{
    dc1394_feature_whitebalance_set_value(camera, position, ui->whiteBalSliderRed->value());
}

void Dc1394Dialog::on_whiteBalSliderRed_sliderMoved(int position)
{
        dc1394_feature_whitebalance_set_value(camera, ui->whiteBalSliderBlue->value(), position);
}

void Dc1394Dialog::on_manualShutter_toggled(bool checked)
{
    CHECK_ERROR(dc1394_feature_set_mode(camera, DC1394_FEATURE_SHUTTER,
                            (checked ? DC1394_FEATURE_MODE_MANUAL : DC1394_FEATURE_MODE_AUTO)));
    //timerEvent(0);
}

void Dc1394Dialog::on_shutterSpin_valueChanged(double val)
{
    CHECK_ERROR(dc1394_feature_set_value(camera, DC1394_FEATURE_SHUTTER, val));
    //timerEvent(0);
}

void Dc1394Dialog::on_autoShutter_clicked()
{
    CHECK_ERROR(dc1394_feature_set_mode(camera, DC1394_FEATURE_SHUTTER, DC1394_FEATURE_MODE_ONE_PUSH_AUTO));
    //timerEvent(0);
}

void Dc1394Dialog::on_manualBrightness_toggled(bool checked)
{
    CHECK_ERROR(dc1394_feature_set_mode(camera, DC1394_FEATURE_BRIGHTNESS,
                            (checked ? DC1394_FEATURE_MODE_MANUAL : DC1394_FEATURE_MODE_AUTO)));
}
